/**********************************************************************************
 Persistence of Vision Ray Tracer Scene Description File
 File name   : HF_Sedimentary_test-test.pov
 Version     : 3.8
 Description : Scene for testing heightfields as landscape sedimentary structures.
               
 Date        : October 2023
 Author      : Thomas de Groot
 
**********************************************************************************/

//+w1280 +h640 +a0.1 +am2 +bm2 +bs8

#version 3.7;

#include "colors.inc"
#include "rand.inc"
#include "functions.inc"
#include "transforms.inc"

#default {pigment {rgb <1,0,0>}}
#default {finish {ambient 0}}

//=================================================================================
// SCENE SWITCHES
//=================================================================================
#declare Rad            = on;		// Radiosity switch

//camera:
#declare CamZoom        = 1;
#declare CamSky         = <0, 1, 0>;
#declare CamAng         = 50;
#declare CamLoc         = vrotate(<0.0, 25.0, -150.0>, <0.0, 30.0, 0.0>);
#declare CamLookAt      = <0.0, 25.0, 0.0>;
#declare UseFocalBlur   = true;
#declare FocalBlurSamples = 10;

//sunlight:
#declare Intensity      = 1.25;
#declare SunCol         = <0.9, 0.9, 0.8>*Intensity;
#declare SunPos         = <0, 0, -2>*10e4;
#declare SunDia         = vlength(SunPos)*2/215;  //apparent diameter of the Sun
#declare Sun_alt 			  = 60;
#declare Sun_azm 			  = -50;

//height_field:
#declare HF_scale       = <150, 5, 50>;
#declare HF_res         = 4000;

//=================================================================================
// GLOBAL SETTINGS
//=================================================================================
global_settings {
  assumed_gamma 1.0
  max_trace_level 5
  #if (Rad)
  radiosity {
    pretrace_start 0.08
    pretrace_end   0.04
    count 50, 1000
    nearest_count 10, 5
    error_bound 1
    recursion_limit 2
    low_error_factor .3
    gray_threshold 0.0
    minimum_reuse 0.015
    maximum_reuse 0.1
    brightness 1

    adc_bailout 0.01/2
    normal off
    media off
    always_sample off
    //max_sample 1.0
  }
  #end
}

//====================================================================================
// CAMERA AND LIGHT
//====================================================================================
camera {
  location  CamLoc
  sky       CamSky  
  up        CamSky
  direction z*CamZoom
  right     x*image_width/image_height
  angle     CamAng
  look_at   CamLookAt
  #if(UseFocalBlur)
    aperture 1 focal_point 0 blur_samples FocalBlurSamples variance 0
  #end
}

light_source {
  <0, 0, 0>
  color srgb SunCol 
  translate SunPos
  rotate Sun_alt*x
  rotate Sun_azm*y
  parallel
  point_at <0, 0, 0>
}

// sky sphere:
sphere { <0, 0, 0>, 1 
  pigment {
    gradient y
    color_map {
      [0.0 srgb <0.6, 0.7, 1.0>]
      [0.7 srgb <0.0, 0.1, 0.8>]
    }
  }
  finish {emission 0.8 diffuse 0}
  scale 5*10e4
  inverse
}

//====================================================================================
// LANDSCAPE ELEMENTS
//====================================================================================
// Simple scene world scale at origin:  8 POV-units = 1.00 m -------------------------
#declare WorldScale =
union {
  cylinder { 
    <0, 0, 0>, <0, 0.5, 0>, 0.025 
    texture { 
      pigment {rgb <1, 0, 0>}
      finish  {diffuse 0.8 specular 0.15 roughness 0.001} 
    }
  }
  cylinder { 
    <0, 0.5, 0>, <0, 1.0, 0>, 0.025 
    texture { 
      pigment {rgb <1, 1, 1>}
      finish  {diffuse 0.8 specular 0.15 roughness 0.001} 
    }
  }
  scale 8
}

#declare Axis =
union {
  object {WorldScale}                 //0-1m
  object {WorldScale translate 8*y}   //1-2m
  object {WorldScale translate 16*y}  //2-3m
}

Axis
object {Axis rotate  90*x}
object {Axis rotate -90*z}

// ground -----------------------------------------------------------------
#declare RasterScale = 8;
#declare RasterHalfLineX = 0.035;  
#declare RasterHalfLineZ = 0.035; 

#macro Raster(RScale, HLine) 
  pigment{ 
    gradient x 
    scale RScale
    color_map {
      [0.000   color rgbt<1,1,1,0>*0.25]
      [0+HLine color rgbt<1,1,1,0>*0.25]
      [0+HLine color rgbt<1,1,1,1>]
      [1-HLine color rgbt<1,1,1,1>]
      [1-HLine color rgbt<1,1,1,0>*0.25]
      [1.000   color rgbt<1,1,1,0>*0.25]
    } 
  }
#end  

// plane with layered textures
plane { 
  <0,1,0>, 0    
  texture { 
    pigment {White*1.1}
    finish {diffuse 0.5}
  }
  texture { Raster(RasterScale,RasterHalfLineX) rotate<0,0,0> }
  texture { Raster(RasterScale,RasterHalfLineZ) rotate<0,90,0>}
  rotate<0,0,0>
}

//====================================================================================
// height_field -----------------------------------------------------------
#debug "\nHeight_field started...\n"

#macro HF_warp()
    warp {repeat x}
    warp {repeat y}
    scale 10
    warp {
      turbulence <0.2, 0.3, 0.4>
      octaves 3 //[6]
      lambda 1  //[2]
      omega 0.2 //[0.5]
    }
    scale 1/10
#end

//-------------------------------------------------------------
#declare F_HF_ =
function {
  pigment {
    image_map {
      png "Sedimentary Layers_32bits_2.png" gamma 1.0
      map_type 0
      interpolate 2
    }
    HF_warp()
  }
}

//-------------------------------------------------------------
#declare HF_ =
  intersection {
    height_field {
      function HF_res, HF_res {F_HF_(x,y,z).hf}
      translate <-0.5, 0.0, -0.5>
      pigment {gradient y pigment_map {[0.0 rgb z][0.5 rgb y][1.0 rgb x+y]}}
      scale HF_scale
    }
    box {
      <-0.5, -50.0, -0.5>, <0.5, 1, 0.5>
      pigment {rgb 0.5}
      scale HF_scale * <0.98, 1, 0.98>
    }
    scale <1/0.98, 1, 1/0.98>
    rotate 180*y
    translate HF_scale.z/2*z
    rotate -90*x    //rotating to vertical position
  } 
  
#debug concat("\nminextent HF_: <",vstr(3, min_extent(HF_), ", ", 0,3),">\n")
#debug concat("maxextent HF_: <",vstr(3, max_extent(HF_), ", ", 0,3),">\n")

//-------------------------------------------------------------------------------------
// matrix structure:
// the first 3 lines form a rotation matrix; since it is not orthogonal, shearing occurs
// the last 3 values contain the translation

intersection {
  object {HF_ 
    matrix <
      1.0, 0.0, 0.0,
      0.0, 1.0, 1.0,
      0.0, 0.0, 2.0,
      0.0, 0.5, 0.0          
    >
  }
  object {HF_ 
    //scale <-1, 1, 1>
    matrix <
      1.0, 0.0, 0.0,
      0.0, 1.0, 0.5,
      0.0, 0.0, 0.5,
      0.0, 0.5, 0.0
    >
    rotate 45*y 
  }
  
}

#debug "\nDone.\n"

//====================================================================================
